#include "control/sync_control.h"

#include <stdlib.h>
#include <libudev.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <glob.h>
#include <alloca.h>


#include "util/logger.h"

#include "model/device_list.h"
#include "model/attribute_list.h"


// udev context and monitor used to receive uevents
static struct udev *udev_context=NULL;
static struct udev_monitor *udev_monitor=NULL;

static int timeout=-1;
struct timespec timeout_start_stamp;

static error_code_t sync_control_process_udevice_added_event(struct udev_device *udevice);
static error_code_t sync_control_apply_creds(const char *path, const kms_credentials_t *creds);
static error_code_t sync_control_check_attributes_filtered(const char *syspath);
static void sync_control_disconnect_udev(void);
static void sync_control_dump_missing(void);
static error_code_t sync_control_apply_attr_creds_callback (const char *path,
		void *data);
static error_code_t sync_control_apply_device_creds_callback (const char *path,
		void *data);
static error_code_t sync_control_path_exist(const char *path, void *data,
		error_code_t(apply_on_match)(const char* matched_path, void *data) );
static error_code_t sync_control_apply_device_credentials(const char *identifier, const kms_credentials_t *creds);
static error_code_t sync_control_apply_creds_of_device_by_syspath(const char *identifier,const kms_credentials_t *creds);
static error_code_t sync_control_connect_to_kernel_socket(void);

//TODO #warning enable late connection to udev on the first call of the add device / attribute functions (to be implemented) that actually puts something into the list
//------------------------------------------ public functions --------------------------------------------------------
void sync_control_init(void)
{
	timeout=-1;
	logger_log_debug("SYNC_CONTROL -> init done.");
}

void sync_control_deinit(bool dump_missing)
{
	if (dump_missing)
		sync_control_dump_missing();

	if (udev_monitor!=NULL)
	{
		udev_monitor_unref(udev_monitor);
		udev_monitor=NULL;
	}

	if (udev_context!=NULL)
	{
		udev_unref(udev_context);
		udev_context=NULL;
	}
	timeout=-1;
	logger_log_debug("SYNC_CONTROL -> deinit done.");
}

int sync_control_get_pollfd(void)
{
	if(udev_monitor!=NULL)
		return udev_monitor_get_fd(udev_monitor);
	else
		return -1;
}

error_code_t sync_control_on_event(void)
{
	error_code_t result=RESULT_OK;
	struct udev_device *udevice;

	if (udev_monitor==NULL)
		return RESULT_INVALID;

	logger_log_debug("SYNC_CONTROL -> Uevent received");
	udevice=udev_monitor_receive_device(udev_monitor);
	if (udevice!=NULL)
	{
		if (strcmp(udev_device_get_action(udevice),"add")==0)
			result=sync_control_process_udevice_added_event(udevice);
		udev_device_unref(udevice);
	}

	return result;
}

int sync_control_get_state(void)
{
	int state;

	state=SYNC_CO_STATE_DONE;

	if (!attribute_list_is_empty())
			state |= SYNC_CO_STATE_WAITING_FOR_ATTRIBUTES;
	if (!device_list_is_empty())
			state |= SYNC_CO_STATE_WAITING_FOR_DEVICES;
	return state;
}

void sync_control_activate_timeout(int timeout_ms)
{
	timeout=timeout_ms;
	logger_log_debug("SYNC_CONTROL - Starting overall timeout.");
	clock_gettime(CLOCK_MONOTONIC,&timeout_start_stamp);
}

int sync_control_get_remaining_timeout_time(void)
{
	struct timespec cur_time;
	int time_passed_ms;
	int remaining_timeout;

	if (timeout==-1)
		return -1;

	clock_gettime(CLOCK_MONOTONIC,&cur_time);
	time_passed_ms=(cur_time.tv_sec-timeout_start_stamp.tv_sec)*1000;
	time_passed_ms+=(cur_time.tv_nsec-timeout_start_stamp.tv_nsec)/1000000;

	remaining_timeout=timeout-time_passed_ms;
	if (remaining_timeout<0)
		remaining_timeout=0;

	return remaining_timeout;
}

error_code_t sync_control_check_attributes(void)
{
	error_code_t result;
	kms_attribute_t *attr;

	result=RESULT_OK;
	attr=attribute_list_get_first_attribute();

	logger_log_debug("SYNC_CONTROL - Checking for attributes.");

	while (attr!=NULL && result==RESULT_OK)
	{
		kms_credentials_t *credentials = &(attr->credentials);
		char* attr_path = attribute_get_attr_path(attr);
		error_code_t tmp_result;

		tmp_result = sync_control_path_exist(attr_path,
				(void*)credentials,
				sync_control_apply_attr_creds_callback);
		if (tmp_result == RESULT_GLOB_NO_MATCHING_PATH)
		{
			attr=attribute_list_get_next_attribute(attr);
		} else {
			if (tmp_result == RESULT_OK) {
				kms_attribute_t *tmp;
				tmp=attr;
				attr=attribute_list_get_next_attribute(attr);
				attribute_list_remove_and_free(tmp);
			} else
				result = tmp_result;
		}
	}

	return RESULT_OK;
}

error_code_t sync_control_add_device_no_credentials(const char *identifier)
{
	return sync_control_add_device(identifier,NULL);
}

error_code_t sync_control_add_device(const char *identifier, const kms_credentials_t *credentials)
{
	error_code_t result=RESULT_OK;
	logger_log_debug("SYNC_CONTROL - Checking device \'%s\'.",identifier);

	//once we receive the first device to wait for and to set credentials, we are starting listening for uevents
	if (udev_monitor==NULL)
		result=sync_control_connect_to_kernel_socket();

	if (result==RESULT_OK)
	{
		result = sync_control_path_exist(identifier, (void*)credentials,
					sync_control_apply_device_creds_callback);
		if (result == RESULT_GLOB_NO_MATCHING_PATH)
		{
			logger_log_debug("SYNC_CONTROL - Device not yet available. Adding %s to the device list.", identifier);
			result=device_list_add(identifier,credentials);
		}
		else if (result == RESULT_UEVENT_FILE_NOT_AVAILABLE)
		{
			logger_log_debug("SYNC_CONTROL - Uevent file not yet available. Adding %s to the device list.", identifier);
			result=device_list_add(identifier,credentials);
		}
		else if (result == RESULT_DEVNODE_NOT_AVAILABLE)
		{
			logger_log_debug("SYNC_CONTROL - Device node not yet available. Adding %s to the device list.", identifier);
			result=device_list_add(identifier,credentials);
		}
	}

	return result;
}

error_code_t sync_control_add_attribute_no_credentials(const char *sysfs_path,
		const char *name)
{
	return sync_control_add_attribute(sysfs_path,name,NULL);
}

error_code_t sync_control_add_attribute(const char *sysfs_path,
		const char *name, const kms_credentials_t *credentials)
{
	error_code_t result;
	kms_attribute_t *attribute;
	char *attr_path;

	result=attribute_list_new_attribute(&attribute, sysfs_path,name,credentials);

	if (result == RESULT_OK) {
		attr_path = attribute_get_attr_path (attribute);

		logger_log_debug("SYNC_CONTROL - Checking attribute \'%s\'.",attr_path);
		result = sync_control_path_exist(attr_path, (void*)credentials,
				sync_control_apply_attr_creds_callback);
		if (result == RESULT_GLOB_NO_MATCHING_PATH)
		{
			logger_log_debug("SYNC_CONTROL - Attribute not yet available. Adding it to the attribute list.");
			result = attribute_list_add(attribute);
		} else {
			attribute_list_free_attribute (attribute);
		}
	}

	return result;
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------------------ private functions -------------------------------------------------------
static error_code_t sync_control_process_udevice_added_event(struct udev_device *udevice)
{
	kms_device_t *device;
	const char *syspath;
	const char *devnode_path;

	// check if device is in our list
	syspath=udev_device_get_syspath(udevice);
	devnode_path=udev_device_get_devnode(udevice);

	device=device_list_get_device(syspath,devnode_path);
	//device not found -> just check for attributes
	if (device==NULL)
		return sync_control_check_attributes_filtered(syspath);

	logger_log_debug("SYNC_CONTROL - Device %s appeared.",syspath);
	logger_log_debug("SYNC_CONTROL - Fits to entry in list: \'%s\'.",device->identifier);

	//device found -> apply creds if needed
	if (device->apply_credentials)
	{
		if (devnode_path==NULL)
		{
			logger_log_error("Unable to set access rights of device %s. The device has no device node.",syspath);
			return RESULT_INVALID_ARGS;
		}
		sync_control_apply_creds(devnode_path,&device->credentials);
	}

	//remove device from list
	device_list_remove_and_free(device);

	//disable udev receiver when we have no device in list anymore
	if (device_list_is_empty())
		sync_control_disconnect_udev();

	// try to find matching attributes to check now
	return sync_control_check_attributes_filtered(syspath);
}

static void sync_control_disconnect_udev(void)
{
	if (udev_monitor!=NULL)
	{
		udev_monitor_unref(udev_monitor);
		udev_monitor=NULL;
	}
}

static error_code_t sync_control_check_attributes_filtered(const char *syspath)
{
	error_code_t result=RESULT_OK;

	kms_attribute_t *attr;
	attr=attribute_list_get_first_attribute_filtered(syspath);
	while (attr!=NULL && result==RESULT_OK)
	{
		kms_credentials_t *credentials = &(attr->credentials);
		char* attr_path = attribute_get_attr_path(attr);
		error_code_t tmp_result;

		tmp_result = sync_control_path_exist(attr_path,
				(void*)credentials,
				sync_control_apply_attr_creds_callback);
		if (tmp_result == RESULT_GLOB_NO_MATCHING_PATH)
		{
			attr=attribute_list_get_next_attribute_filtered(attr,syspath);
		} else {
			if (tmp_result == RESULT_OK) {
				kms_attribute_t *tmp;
				tmp=attr;
				attr=attribute_list_get_next_attribute_filtered(attr,syspath);
				attribute_list_remove_and_free(tmp);
			} else
				result = tmp_result;
		}
	}

	return result;
}

static error_code_t sync_control_apply_creds(const char *path, const kms_credentials_t *creds)
{
	if (creds==NULL)
	{
		return RESULT_OK;
	}

	logger_log_debug("Setting owner and group ids and access mode: uid=%d, gid=%d, mode=%o", creds->user_id,
			creds->group_id, creds->mode);

	if (chown(path,creds->user_id,creds->group_id)==-1)
	{
		logger_log_error("Setting owner on %s failed (uid=%d,gid=%d): %s", path, creds->user_id,
				creds->group_id, strerror(errno));
		return RESULT_APPLYING_CREDS_FAILED;
	}

	if (chmod(path,creds->mode)==-1)
	{
		logger_log_error("Setting mode 0%o of %s failed: %s", creds->mode, path, strerror(errno));
		return RESULT_APPLYING_CREDS_FAILED;
	}

	return RESULT_OK;
}

static void sync_control_dump_missing(void)
{
	kms_attribute_t *attr;
	kms_device_t *device;

	logger_log_error("Following attributes and devices didn't appear:");
	attr=attribute_list_get_first_attribute();
	while (attr!=NULL)
	{
		logger_log_error("\tAttribute: %s",attribute_get_attr_path(attr));
		attr=attribute_list_get_next_attribute(attr);
	}

	device=device_list_get_first_device();
	while (device!=NULL)
	{
		logger_log_error("\tDevice: %s",device->identifier);
		device=device_list_get_next_device(device);
	}
}

static error_code_t sync_control_apply_attr_creds_callback (const char *path,
		void *data) {
	const kms_credentials_t *credentials = (kms_credentials_t*)data;

	logger_log_debug("SYNC_CONTROL - Attribute %s available.\n", path);

	return sync_control_apply_creds(path,credentials);
}

static error_code_t sync_control_apply_device_creds_callback (const char *path,
		void *data) {
	const kms_credentials_t *credentials = (kms_credentials_t*)data;

	return sync_control_apply_device_credentials(path, credentials);
}

static error_code_t sync_control_path_exist(const char *path, void *data,
		error_code_t(apply_on_match)(const char* matched_path, void *data) )
{
	error_code_t result;
	int glob_res;
	glob_t gl;

	glob_res=glob (path,GLOB_NOSORT, NULL, &gl);

	if (glob_res == 0)
	{
		if (gl.gl_pathc > 1) {
			logger_log_error("More than one path matching to the given configuration \'%s\'.", path);
			result = RESULT_GLOB_MULTIPLE_MATCHING_PATHS;
		} else {
			result = apply_on_match(gl.gl_pathv[0], data);
		}
	} else
		if (glob_res == GLOB_NOMATCH)
			result = RESULT_GLOB_NO_MATCHING_PATH;
		else
			result = RESULT_NORESOURCES;

	globfree(&gl);

	return result;
}

static error_code_t sync_control_dev_node_exist(const char *path)
{
	error_code_t result;
	int acc_res;

	acc_res = access(path, F_OK);

	if (acc_res == 0)
	{
		result = RESULT_OK;
	}
	else
	{
		if (errno == ENOENT)
		{
			result = RESULT_DEVNODE_NOT_AVAILABLE;
		}
		else
		{
			logger_log_error("%s", strerror(errno));
			result = RESULT_INVALID;
		}
	}

	return result;
}

static error_code_t sync_control_apply_device_credentials(const char *identifier, const kms_credentials_t *creds)
{
	if (strstr(identifier,"/dev")==identifier)
	{
		//we are a device node
		logger_log_debug("SYNC_CONTROL - Device already available.");
		return sync_control_apply_creds(identifier,creds);

	}
	else
	{
		//we are a syspath
		return sync_control_apply_creds_of_device_by_syspath(identifier,creds);
	}
}

static error_code_t sync_control_apply_creds_of_device_by_syspath(const char *identifier, const kms_credentials_t *creds)
{
	const char *dev_node=NULL;
	struct udev_device *udevice=NULL;
	error_code_t result=RESULT_OK;

	//lazy loading the udev context, we never need it at all in some runs
	if (udev_context==NULL)
		udev_context=udev_new();
	if (udev_context==NULL)
		result=RESULT_NORESOURCES;

	if (result==RESULT_OK)
	{
		udevice=udev_device_new_from_syspath(udev_context, identifier);
		if (udevice==NULL)
		{
			result=RESULT_UEVENT_FILE_NOT_AVAILABLE;
		}
	}

	if (result==RESULT_OK)
	{
		dev_node=udev_device_get_devnode(udevice);
		if (dev_node==NULL)
		{
			if (creds!=NULL)
			{
				logger_log_error("Device \'%s\' has no device node but access rights are defined in the config file.", identifier);
				result=RESULT_INVALID_ARGS;
			}
			else
			{
				logger_log_debug("SYNC_CONTROL - Device \'%s\' has no device node and no access rights are defined.", identifier);
				result=RESULT_OK;
			}
		}
		else
		{
			logger_log_debug("Device \'%s\' has device node \'%s\'", identifier, dev_node);
			result=sync_control_dev_node_exist(dev_node);

			if (result==RESULT_OK)
			{
				logger_log_debug("SYNC_CONTROL - Device already available.");
				result=sync_control_apply_creds(dev_node, creds);
			}
		}
	}

	if (udevice!=NULL)
		udev_device_unref(udevice);

	return result;
}

static error_code_t sync_control_connect_to_kernel_socket(void)
{
	error_code_t result=RESULT_OK;

	//lazy loading the udev context, we never need it at all in some runs
	if (udev_context==NULL)
		udev_context=udev_new();
	if (udev_context==NULL)
		result=RESULT_NORESOURCES;

	if (result==RESULT_OK)
	{
		//we are listening on kernel events since we want to be independent from udevd
		udev_monitor = udev_monitor_new_from_netlink(udev_context, "kernel");
		if (udev_monitor!=NULL)
		{
			if (udev_monitor_enable_receiving(udev_monitor))
				result=RESULT_NORESOURCES;
		}
	}

	return result;
}
//--------------------------------------------------------------------------------------------------------------------
